#!/bin/bash

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
NC='\033[0m' 

# Detect container runtime
detect_container_runtime() {
    if command -v podman &> /dev/null; then
        echo "podman"
    elif command -v docker &> /dev/null; then
        echo "docker"
    else
        echo ""
    fi
}

CONTAINER_RUNTIME=$(detect_container_runtime)

show_spinner() {
    local pid=$1
    local message=$2
    local delay=0.1
    local spinstr='|/-\\'
    echo  " $message "
    while ps -p $pid &>/dev/null; do
        local temp=${spinstr#?}
        printf " [%c] " "$spinstr"
        local spinstr=$temp${spinstr%"$temp"}
        sleep $delay
        printf "\b\b\b\b\b\b"
    done
}

verify_required_tools() {
    local missing_tools=()
    for tool in openssl awk sed cut; do
        if ! command -v "$tool" &> /dev/null; then
            missing_tools+=("$tool")
        fi
    done
    
    if [[ ${#missing_tools[@]} -gt 0 ]]; then
        echo "$RED[ERROR] Required utils are missing: ${missing_tools[*]}. Please install them.$NC" >&2
        exit 1
    fi
    
    echo "$GREEN[OK]$NC All required utils are installed."
}

# Check container runtime installation
check_container_runtime() {
    if [[ -z "$CONTAINER_RUNTIME" ]]; then
        echo -e "$RED[ERROR] Neither Docker nor Podman is installed. Please install one of them first.$NC"
        echo -e "$YELLOW[INFO] You can install Podman or Docker using:$NC"
        echo -e "$YELLOW      - Podman: https://podman.io/getting-started/installation$NC"
        echo -e "$YELLOW      - Docker: https://docs.docker.com/engine/install/$NC"
        return 1
    fi
    echo "$GREEN[OK]$NC $CONTAINER_RUNTIME is installed."
}

# Check compose tool
check_compose_tool() {
    if [[ "$CONTAINER_RUNTIME" == "podman" ]]; then
        # Podman 4.0+ has built-in compose support
        if podman compose version &> /dev/null; then
            echo -e "$GREEN[OK]$NC podman compose in-built utility is used."
            return 0
        # Or check for podman-compose
        elif command -v podman-compose &> /dev/null; then
            echo -e "$GREEN[OK]$NC podman-compose utility is used."
            return 0
        else
            echo -e "$YELLOW[WARN] podman-compose not found. Some features may require it.$NC"
            echo -e "$YELLOW[INFO] You can install it with: 'pip install podman-compose'$NC"
            return 0  # Don't exit, podman itself might be sufficient
        fi
    else
        # Docker case
        if command -v docker-compose &> /dev/null; then
            echo -e "$GREEN[OK]$NC docker-compose utility is used."
        elif docker compose version &> /dev/null; then
            echo -e "$GREEN[OK]$NC docker compose in-built utility is used."
        else
            echo -e "$RED[ERROR] Docker compose utility not found.$NC"
            return 1
        fi
    fi
}

# Check access to Apliqo UX repositories
check_container_auth() {
    echo -n "Checking access to Apliqo UX repositories..."
    
    echo -n "."
    
    local result=0
    if [[ "$CONTAINER_RUNTIME" == "podman" ]]; then
        podman manifest inspect apliqo/spa:latest &> /dev/null
        result=$?
        echo -n "."
        podman manifest inspect apliqo/api:latest &> /dev/null
        [[ $result -eq 0 ]] && result=$?
    else
        docker manifest inspect apliqo/spa:latest &> /dev/null
        result=$?
        echo -n "."
        docker manifest inspect apliqo/api:latest &> /dev/null
        [[ $result -eq 0 ]] && result=$?
    fi
    
    if [ $result -eq 0 ]; then
        echo -e "$GREEN[OK]$NC"
        return 0
    else
        echo -e "$RED[FAILED]$NC"
        echo "Please check your authentication:"
        [[ "$CONTAINER_RUNTIME" == "podman" ]] && echo "  podman login docker.io" || echo "  docker login"
        return 1
    fi
}

# Check license folder
check_license_folder() {
    if [[ -d "license" ]]; then
        echo -e "$GREEN[OK]$NC Folder 'license' exists."
    else
        echo -e "$BLUE[INFO]$NC Folder 'license' doesn't exist."
        mkdir license;
        echo -e "$BLUE[INFO]$NC Folder 'license' created."
    fi
}

# Check styles folder
check_styles_folder() {
    if [[ -d "styles" ]]; then
        echo -e "$GREEN[OK]$NC Folder 'styles' exists."
    else
        echo -e "$BLUE[INFO]$NC Folder 'styles' doesn't exist."
        mkdir styles;
        touch styles/custom.css;
        echo -e "$BLUE[INFO]$NC Folder 'styles' created."
    fi
}

# Check storage folder
check_storage_folder() {
    if [[ -d "storage" ]]; then
        echo -e "$GREEN[OK]$NC Folder 'storage' exists."
    else
        echo -e "$BLUE[INFO]$NC Folder 'storage' doesn't exist."
        mkdir storage;
        echo -e "$BLUE[INFO]$NC Folder 'storage' created."
    fi
}

# Check license mount in docker-compose.yml
check_compose_license_mount() {
    if grep -q "apliqo-api:" docker-compose.yml && grep -q "license:/usr/src/app/license" docker-compose.yml; then
        echo -e "$GREEN[OK]$NC License volume is mounted for 'apliqo-api' in 'docker-compose.yml'."
    else
        echo -e "$RED[ERROR] Please add './license:/usr/src/app/license' volume for 'apliqo-api' in 'docker-compose.yml'.$NC"
        return 1
    fi
}

# Check styles mount in docker-compose.yml
check_compose_styles_mount() {
    if grep -q "apliqo-spa:" docker-compose.yml && grep -q "styles:/usr/src/app/styles" docker-compose.yml; then
        echo -e "$GREEN[OK]$NC Styles volume is mounted for 'apliqo-spa' in 'docker-compose.yml'."
    else
        echo -e "$RED[ERROR] Please add './styles:/usr/src/app/styles' volume for 'apliqo-spa' in 'docker-compose.yml'.$NC"
        return 1
    fi
}

# Check storage mount in docker-compose.yml
check_compose_storage_mount() {
    if grep -q "apliqo-api:" docker-compose.yml && grep -q "storage:/usr/src/app/storage" docker-compose.yml; then
        echo -e "$GREEN[OK]$NC Storage volume is mounted for 'apliqo-api' in 'docker-compose.yml'."
    else
        echo -e "$RED[ERROR] Please add './storage:/usr/src/app/storage' volume for 'apliqo-api' in 'docker-compose.yml'.$NC"
        return 1
    fi
}

# Login to container registry
login_container_registry() {
    echo -e "$BLUE Please set your Access Token below.$NC"
    
    if [[ "$CONTAINER_RUNTIME" == "podman" ]]; then
        echo -e "$YELLOW[INFO] Using Podman runtime$NC"
        podman login -u apliqocustomer || exit 1
    else
        docker login -u apliqocustomer || exit 1
    fi
}

# Logout from container registry
logout_container_registry() {
    if [[ "$CONTAINER_RUNTIME" == "podman" ]]; then
        podman logout
    else
        docker logout
    fi
}

# Display environment information
show_environment_info() {
    echo -e "\n${BLUE}=== Environment Information ===$NC"
    echo -e "Container Runtime: $CONTAINER_RUNTIME"
    
    if [[ "$CONTAINER_RUNTIME" == "podman" ]]; then
        podman --version
        if command -v podman-compose &> /dev/null; then
            podman-compose --version
        fi
    else
        docker --version
        if command -v docker-compose &> /dev/null; then
            docker-compose --version
        elif docker compose version &> /dev/null; then
            docker compose version
        fi
    fi
    echo -e "${BLUE}==============================$NC\n"
}

# Run all checks
run_verification() {
    show_environment_info
    check_container_runtime || exit 1
    check_compose_tool || exit 1
    verify_required_tools || exit 1
    
    check_license_folder || exit 1
    check_compose_license_mount || exit 1
    check_styles_folder || exit 1
    check_storage_folder || exit 1
    check_compose_styles_mount || exit 1
    check_compose_storage_mount || exit 1
    
    echo -e "\n${GREEN}✓ All checks passed successfully!$NC"
    echo -e "${YELLOW}Container runtime: $CONTAINER_RUNTIME$NC"
}

# Main program
main() {
    echo -e "${BLUE}Starting system verification...$NC"
    run_verification
}

# If script is executed directly, run main function
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main
fi