#!/bin/bash

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
NC='\033[0m' 

# 检测容器运行时（如果需要）
detect_container_runtime() {
    if command -v docker &> /dev/null && docker --version | grep -q "Docker"; then
        echo "docker"
    elif command -v podman &> /dev/null && podman --version | grep -q "podman"; then
        echo "podman"
    else
        echo "none"
    fi
}

CONTAINER_RUNTIME=$(detect_container_runtime)

# Generate random string
generate_random_string() {
    openssl rand -hex 16
}

set_container_runtime_env() {
    if [[ "$CONTAINER_RUNTIME" == "podman" ]]; then
        echo -e "$BLUE[INFO]$NC Setting Podman-specific environment variables..."
        
        if command -v docker-compose &> /dev/null; then
            set_env_variable "COMPOSE_CMD" "docker-compose"
        elif podman compose version &> /dev/null; then
            set_env_variable "COMPOSE_CMD" "podman compose"
        fi
        
        set_env_variable "CONTAINER_RUNTIME" "podman"
        set_env_variable "DOCKER_HOST" "unix:///run/user/$(id -u)/podman/podman.sock"
        
    elif [[ "$CONTAINER_RUNTIME" == "docker" ]]; then
        echo -e "$BLUE[INFO]$NC Setting Docker-specific environment variables..."
        
        if command -v docker-compose &> /dev/null; then
            set_env_variable "COMPOSE_CMD" "docker-compose"
        elif docker compose version &> /dev/null; then
            set_env_variable "COMPOSE_CMD" "docker compose"
        fi
        
        set_env_variable "CONTAINER_RUNTIME" "docker"
    fi
}

# Check and creation of .env
ensure_env_file() {
    if [[ ! -f .env ]]; then
        if [[ -f .env.example ]]; then
            cp .env.example .env
            echo -e "$BLUE[INFO]$NC .env is created from .env.example."
            
            if [[ "$OSTYPE" == "darwin"* ]]; then
                sed -i '' "s/\(.*SECRET.*=\).*/\1$(generate_random_string)/" .env
                sed -i '' "s/\(.*PASSWORD.*=\).*/\1$(generate_random_string)/" .env
                sed -i '' "s/\(.*TOKEN.*=\).*/\1$(generate_random_string)/" .env
            else
                sed -i "s/\(.*SECRET.*=\).*/\1$(generate_random_string)/" .env
                sed -i "s/\(.*PASSWORD.*=\).*/\1$(generate_random_string)/" .env
                sed -i "s/\(.*TOKEN.*=\).*/\1$(generate_random_string)/" .env
            fi
            echo -e "$BLUE[INFO]$NC Default secrets been updated."
            
            set_container_runtime_env
            
        else
            echo -e "$ERROR[ERROR] File .env.example is not found.$NC"
            exit 1
        fi
    else
        echo -e "$GREEN[OK]$NC Environment Variables are already initialized."
        
        # 检查是否需要更新容器运行时环境变量
        if ! grep -q "^CONTAINER_RUNTIME=" .env; then
            set_container_runtime_env
        fi
    fi
}

# .env variables management
set_env_variable() {
    local VAR_NAME=$1
    local VAR_VALUE=$2
    if [[ -z "$VAR_NAME" ]]; then
        echo -e "$ERROR[ERROR] Incorrect Usage: set_env_variable <VAR_NAME> [<VAR_VALUE>]$NC"
        echo "If VALUE is not provided, a random value will be generated for SECRET/PASSWORD/TOKEN variables."
        exit 1
    fi
    
    # 如果没有提供值，对于特定变量生成随机值
    if [[ -z "$VAR_VALUE" ]]; then
        case "$VAR_NAME" in
            *SECRET*|*PASSWORD*|*TOKEN*|*KEY*)
                VAR_VALUE=$(generate_random_string)
                echo -e "$BLUE[INFO]$NC Generated random value for $VAR_NAME"
                ;;
            *)
                echo -e "$RED[ERROR]$NC No value provided for $VAR_NAME and it's not a secret type variable.$NC"
                exit 1
                ;;
        esac
    fi
    
    if [[ ! -f .env ]]; then
        echo -e "$YELLOW[WARNING]$NC .env file not found, creating..."
        touch .env
    fi
    
    if grep -q "^$VAR_NAME=" .env; then
        if [[ "$OSTYPE" == "darwin"* ]]; then
            sed -i '' "s/^$VAR_NAME=.*/$VAR_NAME=$VAR_VALUE/" .env
        else
            sed -i "s/^$VAR_NAME=.*/$VAR_NAME=$VAR_VALUE/" .env
        fi
        echo -e "$GREEN[UPDATED]$NC $VAR_NAME=$VAR_VALUE"
    else
        echo "$VAR_NAME=$VAR_VALUE" >> .env
        echo -e "$GREEN[ADDED]$NC $VAR_NAME=$VAR_VALUE"
    fi
}

# 显示当前环境变量
show_env() {
    if [[ ! -f .env ]]; then
        echo -e "$YELLOW[WARNING]$NC .env file not found."
        return 1
    fi
    
    echo -e "$BLUE[INFO]$NC Current environment variables:"
    echo "========================================"
    
    # 显示普通变量
    grep -v -E "^(SECRET|PASSWORD|TOKEN|KEY|PASS)" .env | sort
    
    # 显示敏感变量（隐藏值）
    echo ""
    echo -e "$YELLOW[SENSITIVE VARIABLES - values hidden]$NC"
    grep -E "^(SECRET|PASSWORD|TOKEN|KEY|PASS)" .env | sort | sed 's/=.*/=********/'
    
    echo "========================================"
    echo -e "$BLUE[INFO]$NC Container Runtime: $CONTAINER_RUNTIME"
}

# 重置特定变量的值为随机值
reset_secret() {
    local VAR_NAME=$1
    
    if [[ -z "$VAR_NAME" ]]; then
        echo -e "$RED[ERROR]$NC Please specify variable name to reset.$NC"
        echo "Usage: env-manager reset <VAR_NAME>"
        exit 1
    fi
    
    if ! grep -q "^$VAR_NAME=" .env 2>/dev/null; then
        echo -e "$YELLOW[WARNING]$NC Variable $VAR_NAME not found in .env"
        
        # 检查是否是需要重置的变量类型
        case "$VAR_NAME" in
            *SECRET*|*PASSWORD*|*TOKEN*|*KEY*)
                echo -e "$BLUE[INFO]$NC Creating new secret variable: $VAR_NAME"
                set_env_variable "$VAR_NAME" ""
                ;;
            *)
                echo -e "$RED[ERROR]$NC Variable $VAR_NAME is not a secret type variable.$NC"
                exit 1
                ;;
        esac
    else
        set_env_variable "$VAR_NAME" ""
    fi
}

# 验证 .env 文件
validate_env() {
    if [[ ! -f .env ]]; then
        echo -e "$RED[ERROR]$NC .env file not found.$NC"
        return 1
    fi
    
    local missing_vars=()
    local line_num=1
    
    while IFS= read -r line; do
        # 跳过空行和注释
        [[ -z "$line" ]] && continue
        [[ "$line" =~ ^# ]] && continue
        
        # 检查变量是否有值
        if [[ ! "$line" =~ = ]]; then
            echo -e "$YELLOW[WARNING]$NC Line $line_num: Invalid format: $line"
        elif [[ "$line" =~ =$ ]] || [[ "$line" =~ =\s*$ ]]; then
            local var_name=$(echo "$line" | cut -d'=' -f1)
            missing_vars+=("$var_name")
        fi
        ((line_num++))
    done < .env
    
    if [[ ${#missing_vars[@]} -gt 0 ]]; then
        echo -e "$YELLOW[WARNING]$NC Found variables with empty values:"
        for var in "${missing_vars[@]}"; do
            echo "  - $var"
        done
        echo "You can set values using: apliqoux env-manager set <VAR_NAME> <VALUE>"
        return 1
    else
        echo -e "$GREEN[OK]$NC All environment variables have values."
        return 0
    fi
}

# 函数帮助
help() {
    echo "Environment Variables Manager (Supports Docker & Podman)"
    echo "Usage: apliqoux env-manager <command> [args]"
    echo ""
    echo "Commands:"
    echo "  init                          - Initialize .env file from .env.example"
    echo "  set <VAR_NAME> [<VALUE>]      - Set environment variable value"
    echo "  show                          - Show current environment variables"
    echo "  reset <VAR_NAME>              - Reset secret variable to random value"
    echo "  validate                      - Validate .env file for empty values"
    echo "  help                          - Show this help message"
    echo ""
    echo "Examples:"
    echo "  apliqoux env-manager init                    # Initialize .env file"
    echo "  apliqoux env-manager set DB_PASSWORD mypass  # Set specific value"
    echo "  apliqoux env-manager set API_SECRET          # Generate random secret"
    echo "  apliqoux env-manager show                    # Show all variables"
    echo "  apliqoux env-manager reset API_KEY           # Reset to random value"
    echo "  apliqoux env-manager validate                # Validate .env file"
}

case "$1" in
    init)
        ensure_env_file
        ;;
    set)
        set_env_variable "$2" "$3"
        ;;
    show)
        show_env
        ;;
    reset)
        reset_secret "$2"
        ;;
    validate)
        validate_env
        ;;
    help)
        help
        ;;
    *)
        if [[ -n "$1" ]]; then
            echo -e "$RED[ERROR]$NC Unknown command: $1"
        fi
        help
        exit 1
        ;;
esac