#!/bin/bash

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
NC='\033[0m' 

DB_SERVICES=("apliqo-mongo" "apliqo-redis")
SERVICE_SERVICES=("apliqo-spa" "apliqo-api" "bullboard")

# 检测容器运行时
detect_container_runtime() {
    if command -v docker &> /dev/null && docker --version | grep -q "Docker"; then
        echo "docker"
    elif command -v podman &> /dev/null && podman --version | grep -q "podman"; then
        echo "podman"
    else
        echo "none"
    fi
}

# 检测 compose 工具
detect_compose() {
    local runtime=$1
    
    if [[ "$runtime" == "docker" ]]; then
        if command -v docker-compose &> /dev/null; then
            echo "docker-compose"
        elif docker compose version &> /dev/null; then
            echo "docker compose"
        else
            echo -e "$RED[ERROR] Docker compose utility not found. Please install it.$NC" >&2
            exit 1
        fi
    elif [[ "$runtime" == "podman" ]]; then
        if command -v docker-compose &> /dev/null; then
            echo "docker-compose"
        elif podman compose version &> /dev/null; then
            echo "podman compose"
        else
            echo -e "$RED[ERROR] Podman compose utility not found. Please install it.$NC" >&2
            exit 1
        fi
    else
        echo -e "$RED[ERROR] No container runtime detected.$NC" >&2
        exit 1
    fi
}

get_container_cmd() {
    local runtime=$1
    if [[ "$runtime" == "docker" ]]; then
        echo "docker"
    elif [[ "$runtime" == "podman" ]]; then
        echo "podman"
    else
        echo ""
    fi
}

CONTAINER_RUNTIME=$(detect_container_runtime)
if [[ "$CONTAINER_RUNTIME" == "none" ]]; then
    echo -e "$RED[ERROR] Neither Docker nor Podman is installed.$NC"
    exit 1
fi

COMPOSE_CMD=$(detect_compose "$CONTAINER_RUNTIME")
CONTAINER_CMD=$(get_container_cmd "$CONTAINER_RUNTIME")

echo -e "$BLUE[INFO]$NC Using $CONTAINER_RUNTIME as container runtime"

get_services_by_type() {
    local service_type=$1
    
    case "$service_type" in
        db)
            echo "${DB_SERVICES[@]}"
            ;;
        service)
            echo "${SERVICE_SERVICES[@]}"
            ;;
        all)
            local all_services=("${DB_SERVICES[@]}" "${SERVICE_SERVICES[@]}")
            echo "${all_services[@]}"
            ;;
        *)
            echo "$service_type"
            ;;
    esac
}

validate_service_type() {
    local service_type=$1
    
    case "$service_type" in
        db|service|all)
            return 0
            ;;
        *)
            local all_services=("${DB_SERVICES[@]}" "${SERVICE_SERVICES[@]}")
            for service in "${all_services[@]}"; do
                if [[ "$service" == "$service_type" ]]; then
                    return 0
                fi
            done
            
            echo -e "$RED[ERROR] Unknown service type: $service_type$NC"
            echo "Available options: db, service, all, or specific service name"
            return 1
            ;;
    esac
}

verify_docker_compose_file() {
    if [[ ! -f docker-compose.yml ]]; then
        echo "$RED[ERROR] docker-compose.yml not found." >&2
        exit 1
    fi
    
    local inside_services=0
    local missing_containers=()
    local current_service=""
    local has_container_name=0
    
    while IFS= read -r line; do
        # Detect services section
        if [[ $line =~ ^services: ]]; then
            inside_services=1
            continue
        fi
        
        # Break if services section is ended
        if [[ $inside_services -eq 1 && $line =~ ^[a-zA-Z0-9_-]+: ]]; then
            break
        fi
        
        # Detect service name
        if [[ $inside_services -eq 1 ]]; then
            if [[ $line =~ ^[[:space:]]{2}([a-zA-Z0-9_-]+):$ ]]; then
                if [[ $has_container_name -eq 0 && -n $current_service ]]; then
                    missing_containers+=("$current_service")
                fi
                current_service="${BASH_REMATCH[1]}"
                has_container_name=0
            elif [[ $line =~ ^[[:space:]]{4}container_name: ]]; then
                has_container_name=1
            fi
        fi
    done < docker-compose.yml
    
    if [[ $has_container_name -eq 0 && -n $current_service ]]; then
        missing_containers+=("$current_service")
    fi
    
    if [[ ${#missing_containers[@]} -gt 0 ]]; then
        echo -e "$RED[ERROR] 'container_name:' is missing for containers: ${missing_containers[*]}" >&2
        echo -e "$YELLOW Please add 'container_name' property for all containers in docker-compose.yml. The best scenario is to use service name for container_name value$NC" >&2
        exit 1
    fi
    
    echo -e "$GREEN[OK] docker-compose.yml is valid. $NC"
}

# Get list of containers from docker-compose.yml
get_containers() {
    verify_docker_compose_file || exit 1
    local containers=$(awk '!/^[[:space:]]*#/ && /^[[:space:]]*container_name:/ {print $2}' docker-compose.yml)
    if [[ -z "$containers" || "$containers" == "container_name:" ]]; then
        exit 1
    fi
    echo "$containers"
}

# Get container used image name
get_image_name() {
    local container=$1
    grep -A 5 "^  $container:" docker-compose.yml | grep "image:" | awk '!/^[[:space:]]*#/ {print $2}'
}

# Check that container is part of docker-compose.yml
validate_container() {
    local container=$1
    if [[ -n "$container" && ! $(get_containers | grep -w "$container") ]]; then
        echo -e "$RED [ERROR] Container '$container' not found in docker-compose.yml.$NC" >&2
        exit 1
    fi
}

start_services_by_type() {
    local service_type=$1
    
    validate_service_type "$service_type" || return 1
    
    local services_to_start=($(get_services_by_type "$service_type"))
    
    if [[ ${#services_to_start[@]} -eq 0 ]]; then
        echo -e "$YELLOW[WARNING] No services found for type: $service_type$NC"
        return 0
    fi
    
    echo -e "$BLUE[INFO]$NC Starting services for type: $service_type"
    
    for service in "${services_to_start[@]}"; do
        if get_containers | grep -q -w "$service"; then
            echo -e "$BLUE[INFO]$NC Starting service: $service"
            $COMPOSE_CMD up -d "$service" || {
                echo -e "$YELLOW[WARNING]$NC Failed to start $service, continuing..."
            }
        else
            echo -e "$YELLOW[WARNING]$NC Service $service not found in docker-compose.yml, skipping..."
        fi
    done
    
    echo -e "$GREEN[SUCCESS]$NC Services started for type: $service_type"
}

# Run container(s)
start_containers() {
    local container=$1
    
    # 如果没有指定容器，启动所有
    if [[ -z "$container" ]]; then
        echo -e "$BLUE[INFO]$NC Starting all containers..."
        $COMPOSE_CMD up -d
        return
    fi
    
    # 检查是否是服务类型
    case "$container" in
        db|service|all)
            start_services_by_type "$container"
            ;;
        *)
            # 否则视为具体容器名
            validate_container "$container"
            echo -e "$BLUE[INFO]$NC Starting container: $container"
            $COMPOSE_CMD up -d "$container"
            ;;
    esac
}

# Restart container(s)
restart_containers() {
    local container=$1
    validate_container "$container"
    if [[ -n "$container" ]]; then
        $COMPOSE_CMD up -d --force-recreate "$container"
    else
        $COMPOSE_CMD up -d --force-recreate 
    fi
}

# Stop container(s)
stop_containers() {
    local container=$1
    if [[ -z "$container" ]]; then
        $COMPOSE_CMD down
        return
    fi
    
    # 检查是否是服务类型
    case "$container" in
        db|service|all)
            local services_to_stop=($(get_services_by_type "$container"))
            for service in "${services_to_stop[@]}"; do
                if get_containers | grep -q -w "$service"; then
                    echo -e "$BLUE[INFO]$NC Stopping service: $service"
                    $COMPOSE_CMD stop "$service" || true
                fi
            done
            ;;
        *)
            validate_container "$container"
            $COMPOSE_CMD stop "$container"
            ;;
    esac
}

# Container Logs
logs_containers() {
    local container=$1
    validate_container "$container"
    if [[ -n "$container" ]]; then
        $CONTAINER_CMD logs -f "$container"
    else
        echo -e "$RED [ERROR] Please set the container for logs.$NC";
    fi
}

# Service Status
status_containers() {
    if [[ "$CONTAINER_RUNTIME" == "docker" ]]; then
        $COMPOSE_CMD stats
    else
        # Podman 没有 stats 命令，使用替代方案
        echo -e "$BLUE[INFO]$NC Showing container status..."
        $CONTAINER_CMD ps --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    fi
}

# Image Update for container
update_container() {
    local container=$1
    validate_container "$container"
    local image_name=$(get_image_name "$container")
    if [[ -z "$image_name" ]]; then
        echo -e "$RED [ERROR] Cannot found and image for container '$container'.$NC" >&2
        exit 1
    fi
    echo -e "$BLUE[INFO]$NC Image update $image_name..."
    $CONTAINER_CMD pull "$image_name"
}

# Ensure network for containers
ensure_network() {
    if ! $CONTAINER_CMD network ls | grep -q "apliqo-core"; then
        echo -e "$BLUE[INFO]$NC $CONTAINER_RUNTIME network creation..."
        $CONTAINER_CMD network create apliqo-core
    fi
}

# 检查容器运行时命令是否可用
check_container_command() {
    if ! command -v $CONTAINER_CMD &> /dev/null; then
        echo -e "$RED[ERROR] $CONTAINER_CMD command not found.$NC"
        exit 1
    fi
}

# 初始化系统（带服务类型参数）
init_system() {
    local service_type=$1
    
    # 检查容器命令是否可用
    check_container_command
    
    verify_docker_compose_file || exit 1
    ensure_network
    
    if [[ -n "$service_type" ]]; then
        start_services_by_type "$service_type"
    else
        start_containers
    fi
}

# 显示服务分组信息
show_service_groups() {
    echo "Service Groups:"
    echo "  db      - Database services: ${DB_SERVICES[*]}"
    echo "  service - Application services: ${SERVICE_SERVICES[*]}"
    echo "  all     - All services"
}

# 显示容器运行时信息
show_runtime_info() {
    echo "Container Runtime: $CONTAINER_RUNTIME"
    echo "Compose Command: $COMPOSE_CMD"
    echo "Container Command: $CONTAINER_CMD"
}

help() {
    echo "Container Management Script (Supports Docker & Podman)"
    echo "Usage: apliqoux service <command> [args]"
    echo ""
    show_runtime_info
    echo ""
    echo "Commands:"
    echo "  start <container|type>        - Run container(s) or service group"
    echo "  restart <container>           - Restart container(s)"
    echo "  stop <container|type>         - Stop container(s) or service group"
    echo "  logs <container>              - Show logs of container"
    echo "  containers                    - List all containers"
    echo "  status                        - Shows status of containers"
    echo "  update <container>            - Update container image"
    echo "  init [type]                   - Initialize system with optional service type"
    echo ""
    show_service_groups
    echo ""
    echo "Examples:"
    echo "  apliqoux service start db        # Start only database services"
    echo "  apliqoux service start service   # Start only application services"
    echo "  apliqoux service start all       # Start all services"
    echo "  apliqoux service start mongodb   # Start specific service"
    echo "  apliqoux service init db         # Initialize with only databases"
}

case "$1" in
    start)
        start_containers "$2"
        ;;
    restart)
        restart_containers "$2"
        ;;
    stop)
        stop_containers "$2"
        ;;
    logs)
        logs_containers "$2"
        ;;
    containers)
        get_containers
        ;;
    status)
        status_containers
        ;;
    update)
        update_container "$2"
        ;;
    init)
        init_system "$2"
        ;;
    help)
        help
        ;;
    *)
        echo -e "$RED[ERROR]$NC Unknown command: $1"
        help
        exit 1
        ;;
esac