#!/bin/bash
VERSION="0.2.0"

# Scripts
DOCKER_MANAGER="./scripts/docker-scripts.sh"
ENV_MANAGER="./scripts/env-manager.sh"
VERIFY_SYSTEM="./scripts/verify-system.sh"
APLIQOUX_PATH="/usr/local/bin/apliqoux"

DB_SERVICES=("apliqo-mongo" "apliqo-redis")
SERVICE_SERVICES=("apliqo-spa" "apliqo-api" "bullboard")

detect_container_runtime() {
    if command -v docker &> /dev/null && docker --version | grep -q "Docker"; then
        echo "docker"
    elif command -v podman &> /dev/null && podman --version | grep -q "podman"; then
        echo "podman"
    else
        echo "none"
    fi
}

CONTAINER_RUNTIME=$(detect_container_runtime)

if [[ "$CONTAINER_RUNTIME" == "podman" ]]; then
    DOCKER_MANAGER="./scripts/docker-scripts.sh"
fi

source "$VERIFY_SYSTEM";

validate() {
    echo -e "$BLUE[INFO]$NC Validation..."
    
    if [[ "$CONTAINER_RUNTIME" == "none" ]]; then
        echo -e "$RED[ERROR]$NC Neither Docker nor Podman is installed. Please install one of them."
        exit 1
    fi
    
    echo -e "$GREEN[OK]$NC Using $CONTAINER_RUNTIME as container runtime."
    
    run_verification
    bash "$DOCKER_MANAGER" containers || exit 1
    echo -e "$GREEN[SUCCESS]$NC Validation Complete."
}

install_services() {
    local service_type=$1
    local services_to_start=()
    
    case "$service_type" in
        db)
            services_to_start=("${DB_SERVICES[@]}")
            echo -e "$BLUE[INFO]$NC Installing database services (MongoDB and Redis)..."
            ;;
        service)
            services_to_start=("${SERVICE_SERVICES[@]}")
            echo -e "$BLUE[INFO]$NC Installing application services (SPA, API, bullboard)..."
            ;;
        all)
            services_to_start=("${DB_SERVICES[@]}" "${SERVICE_SERVICES[@]}")
            echo -e "$BLUE[INFO]$NC Installing all services..."
            ;;
        *)
            echo -e "$RED[ERROR]$NC Unknown service type: $service_type"
            echo "Available options: db, service, all"
            exit 1
            ;;
    esac
    
    validate || exit 1
    
    bash "$ENV_MANAGER" init || exit 1
    
    echo -e "$BLUE[INFO]$NC Starting services..."
    
    for service in "${services_to_start[@]}"; do
        echo -e "$BLUE[INFO]$NC Starting $service..."
        bash "$DOCKER_MANAGER" start "$service" || {
            echo -e "$YELLOW[WARNING]$NC Failed to start $service, continuing..."
        }
    done
    
    echo -e "$GREEN[SUCCESS]$NC Installation Complete for $service_type services."
}

install() {
    local service_type=$1
    
    if [[ -z "$service_type" ]]; then
        echo -e "$BLUE[INFO]$NC No service type specified, installing all services..."
        service_type="all"
    fi
    
    install_services "$service_type"
}

update_images() {
    validate || exit 1

    local updated=0
    
    local services=()
    if [[ "$CONTAINER_RUNTIME" == "docker" ]]; then
        services=("apliqo-spa" "apliqo-api")
    else
        services=("apliqo-spa" "apliqo-api")
    fi
    
    for service in "${services[@]}"; do
        local image=$(docker compose config | awk -v svc="$service" '$1 == svc":" {found=1} found && /image:/ {print $2; exit}')
        
        if [[ -n "$image" ]]; then
            echo -e "$BLUE[INFO]$NC Checking updates for $image..."
            local local_digest=$(docker inspect --format='{{index .RepoDigests 0}}' "$image" 2>/dev/null | cut -d'@' -f2)
            docker pull "$image" >/dev/null 2>&1
            local remote_digest=$(docker inspect --format='{{index .RepoDigests 0}}' "$image" 2>/dev/null | cut -d'@' -f2)
            
            if [[ "$local_digest" != "$remote_digest" ]]; then
                echo -e "$BLUE[INFO]$NC New image found $image. Restarting container $service..."
                docker compose up -d --no-deps --force-recreate "$service"
                updated=1
            fi
        fi
    done
    
    if [[ $updated -eq 0 ]]; then
        echo -e "$GREEN[OK]$NC Everything is up to date."
    fi
}

expose() {
    echo -e "$BLUE[INFO]$NC Installing apliqoux as global system util..."
    chmod +x "$0"
    sudo cp "$0" "$APLIQOUX_PATH"
    echo -e "$BLUE[INFO]$NC Now 'apliqoux' is available as a global command."
    echo "Try 'apliqoux help' for more information."
}

show_install_help() {
    echo "Installation Options:"
    echo "  apliqoux install              - Install all services (default)"
    echo "  apliqoux install db           - Install only database services (MongoDB, Redis)"
    echo "  apliqoux install service      - Install only application services (SPA, API, Nginx)"
    echo "  apliqoux install all          - Install all services (same as 'apliqoux install')"
}

help() {
    echo "ApliqoUX CLI $VERSION"
    echo "Container Runtime: $CONTAINER_RUNTIME"
    echo "Usage: apliqoux <module> <command> [args]"
    echo "Modules/Commands:"
    echo "  env-manager             - Module: Environment Variables Manager"
    echo "  service                 - Module: Docker Containers Service"
    echo "  expose                  - Command: Exposes apliqoux as a global command"
    echo "  install [type]          - Command: ApliqoUX installation pipeline"
    show_install_help
    echo "  update                  - Command: Update the ApliqoUX images"
    echo "  validate                - Command: Check the system for readiness"
    echo ""
    echo "Examples:"
    echo "  apliqoux install db           # Install only databases"
    echo "  apliqoux install service      # Install only application services"
    echo "  apliqoux install              # Install everything"
}

proxy_command() {
    local module=$1
    shift
    
    case "$module" in
        install)
            install "$@"
            ;;
        update)
            update_images
            ;;
        validate)
            validate
            ;;
        expose)
            expose
            exit 0;
            ;;
        help)
            help
            ;;
        env-manager)
            bash "$ENV_MANAGER" "$@"
            ;;
        service)
            bash "$DOCKER_MANAGER" "$@"
            ;;
        *)
            if [[ -n "$module" ]]; then
                echo -e "$RED[ERROR]$NC Unknown module: $module"
            fi

            help
            exit 1
            ;;
    esac
}

proxy_command "$@"